#!/usr/bin/perl

use strict;

#
#Sample script to demonstrate the use of the RPF Web-Service.
#It is important that the file attachments have the correct Content-Ids.
#
#accepted Ids are:
# bmrbfile, pdbfile, n15noesy, c13aro, c13ali, c13noesy, ccnoesy, and noesy2d
#
#required parameters are:
# email, protein_name, pdbfile, and bmrbfile
#
#optional parameters are:
# [experiment name]_water for the water suppression range
# [experiment name]_d20 if the experiment is in d20
#

use SOAP::Lite;
use MIME::Entity;
use SOAP::Lite::Packager;
use Getopt::Long;

my ($base_path, $email_string, $pdb_id, $info_file);
GetOptions( "b=s" => \$base_path,
            "e=s" => \$email_string,
            "p=s" => \$pdb_id,
            "i=s" => \$info_file,
          );

# *** SUBROUTINES ***
sub build_entity {
   my ($filepath, $id, $filename) = @_;
   return build MIME::Entity
      Type => "text/plain",
      Path => $filepath,
      Filename => $filename,
      Disposition => "attachment",
      Id         => $id;
}
# *******************

my ($PDB, $bmb, $n15nsy, $c13ao, $c13ai, $c13nsy, $ccnsy, $nsy2d, $email, $protein_name, $bmrb, $pdb);
my %info_data;
my @files;

if (! $info_file && (! $base_path || ! $email_string || ! $pdb_id)) {
   print STDERR<<EOF;

      Usage: rpf.pl (-b <base path> -e <email> -p <PDB ID>)|(-i <info file>)

EOF
   exit();
}

chomp($info_file, $base_path, $email_string, $pdb_id);

if (! $info_file) {
   $base_path =~ s/\/$//;

   $PDB    = $base_path . "/$pdb_id.pdb";           # pdbfile
   $bmb    = $base_path . "/$pdb_id.bmrb";          # bmrbfile
   $n15nsy = $base_path . "/$pdb_id.n15nsy.peaks";  # n15noesy
   $c13ao  = $base_path . "/$pdb_id.c13aro.peaks";  # c13aro
   $c13ai  = $base_path . "/$pdb_id.c13ali.peaks";  # c13ali
   $c13nsy = $base_path . "/$pdb_id.c13nsy.peaks";  # c13noesy
   $ccnsy  = $base_path . "/$pdb_id.cc13.peaks";    # ccnoesy
   $nsy2d  = $base_path . "/$pdb_id.nsy2d.peaks";   # noesy2d

   $email        = SOAP::Data->name(email => "$email_string")->type("string");
   $protein_name = SOAP::Data->name(protein_name => $pdb_id)->type("string");

   @files = (
              [$n15nsy, "n15noesy", "N15.peaks"],
              [$c13ao, "c13aro", "C13aro.peaks"],
              [$c13ai, "c13ali", "C13ali.peaks"],
              [$c13nsy, "c13noesy", "C13.peaks"],
              [$ccnsy, "ccnoesy", "CC.peaks"],
              [$nsy2d, "noesy2d", "2D.peaks"],
              [$bmb, "bmrbfile", "BMRB.bmrb"],
              [$PDB, "pdbfile", "PROT.pdb"],
            );
}
else {
   foreach my $key qw(pdb prot_name n15noesy c13aro c13ali c13noesy ccnoesy noesy2d bmrb email) {
      $info_data{$key} = ' ';
   }

   open INFO, "<$info_file" or die "Cannot read from RPF info file $info_file: $!";
   while (<INFO>) {
      if (/(\w+)\s+(.+)/) {
         my $key = lc($1); # first convert to lowercase

         if ($info_data{$key} eq ' ') {
            $info_data{$key} = $2;
         }
         elsif (defined $info_data{$key} && $info_data{$key} ne ' ') {
            print "*** Warning: $1 was provided several times! ***\n";
         }
         elsif (! defined $info_data{$key}) {
            print "*** Warning: $1 is not a valid key -> ignored! ***\n";
         }
      }
   }
   close INFO;

   my $error;
   foreach my $key qw(pdb bmrb email prot_name) {
      if ($info_data{$key} eq ' ') {
         print "*** Error: $key was not provided!";
         $error = 1;
      }
   }

   if ($error == 1) {
      print "   EXITING the RPF calculation\n";
      exit();
   }

   $email        = SOAP::Data->name(email => $info_data{"email"})->type("string");
   $protein_name = SOAP::Data->name(protein_name => $info_data{"prot_name"})->type("string");

   @files = (
              [$info_data{"n15noesy"}, "n15noesy", "N15.peaks"],
              [$info_data{"c13aro"}, "c13aro", "C13aro.peaks"],
              [$info_data{"c13ali"}, "c13ali", "C13ali.peaks"],
              [$info_data{"c13noesy"}, "c13noesy", "C13.peaks"],
              [$info_data{"ccnoesy"}, "ccnoesy", "CC.peaks"],
              [$info_data{"noesy2d"}, "noesy2d", "2D.peaks"],
              [$info_data{"bmrb"}, "bmrbfile", "BMRB.bmrb"],
              [$info_data{"pdb"}, "pdbfile", "PROT.pdb"],
            );
}

my $num_files = $#files + 1;
my @entities;

foreach (0..$num_files-1) {
   if (-e ${$files[$_]}[0]) {
      push @entities, &build_entity(@{$files[$_]});
   }
}

#my $c13ali_water = SOAP::Data->name("c13ali_water" => "4.6-4.9")->type("string");      #water suppression range
#my $c13ali_d20   = SOAP::Data->name("c13ali_d20");      #d20 solution

my $result = SOAP::Lite
   -> service('http://nmr.cabm.rutgers.edu:80/rpf/RpfService.wsdl')
   -> getRpfScoreWithOptions($email, $protein_name, $bmrb, $pdb, @entities);
print $result . "\n";
